/*:
 * @target MZ
 * @plugindesc Comprehensive Keyboard Mapping for RPG Maker MZ. Support Keyboard & Gamepad.
 * @author Sang Hendrix
 * @url https://sanghendrix.itch.io/
 * 
 * @help
 * Verion 1.0.2
 * ==============================================================================
 * This is a plugin for RPG Maker MZ that allow you to assign the default RPG Maker MZ 
 * buttons AND new buttons to Keyboard & Gamepad.
 *==============================================================================
 * CHANGELOG
 * [ver 1.0.2] Parallel common event bug fixes
 * [ver 1.0.1] Fixed an issue that made the keys didn't assign to your game.
 * [ver 1.0.0] Released!
 *==============================================================================
* FEATURES
 * - Assign default RPG Maker MZ functional buttons to a new layout for both
 * Keyboard and Gamepad.
 * - Directly type in any letter on Keyboard. No more typing Javascript numbers,
 * ain't nobody got time for that.
 * - Run a common event with any keyboard/gamepad button that you wish.
 *==============================================================================
 * How to use:
 * Assign new button to default RPG Maker MZ buttons (ok, cancel, shift, etc) by
 * going to parameters Keyboard, Gamepad. Type in any letter you want to Keyboard
 * and the same letter to Gamepad. Your Gamepad will acknowledge your assigned letter.
 * For Example: You assign button f on keyboard for function Cancel. Now when you
 * press f, it'll trigger Cancel function.
 * In Gamepad, at button A, you also assign f to that, it'll automatically understand
 * it's Cancel due to what you assigned to parameter Keyboard.
 * 
 * If you want to create a newly button like pressing m to trigger a common event,
 * you can do so in ExtraButtons parameter. Then, in Gamepad parameter, assign the same
 * letter to any button you wish.
 * ==============================================================================
 * For feedback or feature request, please dm me on X (Twitter):
 * https://x.com/sanghendrix96
 *
 * @param Keyboard
 * @type struct<Keyboard>
 * @default {"Tab":"tab","Ok/Confirm":"e","Control":"ctrl","Escape":"esc","Page Up":"pageup","Page Down":"pagedown","Left":"left","Up":"up","Right":"right","Down":"down","Shift":"shift"}
 *
 * @param Gamepad
 * @type struct<Gamepad>
 * @default {"A":"u","B":"i","X":"o","Y":"p","LB":"q","RB":"w","LT":"e","RT":"r","Back":"t","Start":"y","LS-Press":"a","RS-Press":"s"}
 * 
 * @param Extra Buttons
 * @type struct<ExtraButtons>[]
 * @default []
 * 
 */

/*~struct~Keyboard:
 * @param Tab
 * @desc Character key for the Tab
 * @default tab
 *
 * @param Ok
 * @desc Character key for the Ok/Confirm
 * @default e
 *
 * @param Control
 * @desc Character key for the Control
 * @default ctrl
 *
 * @param Escape
 * @desc Character key for the Escape
 * @default esc
 *
 * @param Page Up
 * @desc Character key for the Page Up
 * @default pageup
 *
 * @param Page Down
 * @desc Character key for the Page Down
 * @default pagedown
 *
 * @param Left
 * @desc Character key for the Left Arrow
 * @default left
 *
 * @param Up
 * @desc Character key for the Up Arrow
 * @default up
 *
 * @param Right
 * @desc Character key for the Right Arrow
 * @default right
 *
 * @param Down
 * @desc Character key for the Down Arrow
 * @default down
 *
 * @param Shift
 * @desc Character key for the Shift
 * @default shift
 */
/*~struct~Gamepad:
 * @param A
 * @desc Gamepad button A
 * @default u
 *
 * @param B
 * @desc Gamepad button B
 * @default i
 *
 * @param X
 * @desc Gamepad button X
 * @default o
 *
 * @param Y
 * @desc Gamepad button Y
 * @default p
 *
 * @param LB
 * @desc Gamepad button LB
 * @default q
 *
 * @param RB
 * @desc Gamepad button RB
 * @default w
 *
 * @param LT
 * @desc Gamepad button LT
 * @default e
 *
 * @param RT
 * @desc Gamepad button RT
 * @default r
 *
 * @param Back
 * @desc Gamepad button Back
 * @default t
 *
 * @param Start
 * @desc Gamepad button Start
 * @default y
 *
 * @param LS-Press
 * @desc Gamepad button LS-Press
 * @default a
 *
 * @param RS-Press
 * @desc Gamepad button RS-Press
 * @default s
 */

/*~struct~ExtraButtons:
 * @param Keyboard Button
 * @desc The keyboard button to trigger the common event.
 * @type string
 * @default
 *
 * @param Common Event ID
 * @desc The ID of the common event to be triggered.
 * @type common_event
 * @default
 *
 * @param Options
 * @desc Choose whether the common event is triggered on key press or key release.
 * @type select
 * @option Triggered
 * @value triggered
 * @option Pressed
 * @value pressed
 * @default triggered
 * 
 * @param AllowParallel
 * @text Allow to run in parallel mode
 * @type boolean
 * @default false
 * @desc If true, allows this common event to be triggered even if another is already running.
 */

var GamepadButtons = {
    'A': 0,
    'B': 1,
    'X': 2,
    'Y': 3,
    'LB': 4,
    'RB': 5,
    'LT': 6,
    'RT': 7,
    'Back': 8,
    'Start': 9,
    'LS-Press': 10,
    'RS-Press': 11
};

(function () {
    var parameters = PluginManager.parameters('Hendrix_Keyboard_Gamepad');
    var keyboardMapping = JSON.parse(parameters['Keyboard']);
    var gamepadMapping = JSON.parse(parameters['Gamepad']);
    var extraButtons = JSON.parse(parameters['Extra Buttons'] || '[]').map(JSON.parse);

    var charToKeyCode = {
        'backspace': 8, 'tab': 9, 'enter': 13, 'shift': 16, 'ctrl': 17, 'alt': 18, 'pause': 19, 'capslock': 20,
        'esc': 27, 'space': 32, 'pageup': 33, 'pagedown': 34, 'end': 35, 'home': 36, 'left': 37, 'up': 38,
        'right': 39, 'down': 40, 'insert': 45, 'delete': 46, '0': 48, '1': 49, '2': 50, '3': 51, '4': 52,
        '5': 53, '6': 54, '7': 55, '8': 56, '9': 57, 'a': 65, 'b': 66, 'c': 67, 'd': 68, 'e': 69, 'f': 70,
        'g': 71, 'h': 72, 'i': 73, 'j': 74, 'k': 75, 'l': 76, 'm': 77, 'n': 78, 'o': 79, 'p': 80, 'q': 81,
        'r': 82, 's': 83, 't': 84, 'u': 85, 'v': 86, 'w': 87, 'x': 88, 'y': 89, 'z': 90, 'windows': 91,
        'numpad0': 96, 'numpad1': 97, 'numpad2': 98, 'numpad3': 99, 'numpad4': 100, 'numpad5': 101,
        'numpad6': 102, 'numpad7': 103, 'numpad8': 104, 'numpad9': 105, 'multiply': 106, 'add': 107,
        'subtract': 109, 'decimalpoint': 110, 'divide': 111, 'f1': 112, 'f2': 113, 'f3': 114, 'f4': 115,
        'f5': 116, 'f6': 117, 'f7': 118, 'f8': 119, 'f9': 120, 'f10': 121, 'f11': 122, 'f12': 123,
        'numlock': 144, 'scrolllock': 145, 'semicolon': 186, 'equals': 187, 'comma': 188, 'dash': 189,
        'period': 190, 'forwardslash': 191, 'graveaccent': 192, 'openbracket': 219, 'backslash': 220,
        'closebracket': 221, 'singlequote': 222, 66: 'C',
    };

    function remapKeys() {
        Input.keyMapper = {};
        Input.gamepadMapper = { 12: 'up', 13: 'down', 14: 'left', 15: 'right' };

        for (var char in charToKeyCode) {
            if (charToKeyCode.hasOwnProperty(char)) {
                var keyCode = charToKeyCode[char];
                if (typeof keyCode === 'number') {
                    Input.keyMapper[keyCode] = char;
                }
            }
        }

        Object.keys(keyboardMapping).forEach(function (key) {
            var action = key.replace(/[^a-zA-Z]/g, '').toLowerCase();
            var charKey = keyboardMapping[key].toLowerCase();
            if (charToKeyCode[charKey]) {
                Input.keyMapper[charToKeyCode[charKey]] = action;
            }
        });

        Object.keys(gamepadMapping).forEach(function (button) {
            var keyAction = gamepadMapping[button].toLowerCase();
            if (charToKeyCode[keyAction]) {
                Input.gamepadMapper[GamepadButtons[button]] = Input.keyMapper[charToKeyCode[keyAction]];
            }
        });

    }

    function triggerCommonEvent(option, key, commonEventId, allowParallel, sceneMap) {
        if ((option === 'triggered' && Input.isTriggered(key)) || (option === 'pressed' && Input.isPressed(key))) {
            if (allowParallel) {
                executeCommonEvent(commonEventId);
            } else if (!sceneMap.isEventRunning()) {
                $gameTemp.reserveCommonEvent(commonEventId);
            }
        }
    }

    function checkExtraButtons() {
        extraButtons.forEach(function (button) {
            var key = button['Keyboard Button'].toLowerCase();
            var commonEventId = Number(button['Common Event ID']);
            var option = button['Options'];
            var allowParallel = button['AllowParallel'] === 'true';
            var sceneMap = SceneManager._scene;
        
            triggerCommonEvent(option, key, commonEventId, allowParallel, sceneMap);
        });
    
    }

    function executeCommonEvent(eventId) {
        const commonEvent = $dataCommonEvents[eventId];
        if (commonEvent) {
            const interpreter = new Game_Interpreter();
            interpreter.setup(commonEvent.list);
            
            if (!$gameTemp._parallelInterpreters) {
                $gameTemp._parallelInterpreters = [];
            }
            
            $gameTemp._parallelInterpreters.push(interpreter);
        }
    }

    Scene_Map.prototype.isEventRunning = function () {
        return $gameMap.isEventRunning() || $gameMessage.isBusy();
    };

   // Modify Scene_Map.prototype.update to include our new update method
   const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this.isActive()) {
            checkExtraButtons();
            if ($gameTemp._parallelInterpreters) {
                $gameTemp._parallelInterpreters = $gameTemp._parallelInterpreters.filter(interpreter => {
                    if (interpreter.isRunning()) {
                        interpreter.update();
                        return true;
                    }
                    return false;
                });
            }
        }
    };

    remapKeys();
})();
